-- FUNCTIONS (delimited with EOF)
DELIMITER EOF

-- Add a to the given timestamp the given interval
-- Arguments:
--   $1: The timestamp
--   $2: The amount to add / subtract (when negative)
--   $3: The field (HOURS, DAYS, etc). The name is case insensitive and can be in singular or plural (e.g. month, MONTHS are the same)
-- Returns: The new timestamp
create or replace function cy_add_interval 
    (timestamp with time zone, integer, varchar) 
    returns timestamp with time zone 
    as $$ select $1 + cast(concat($2, ' ', $3) as interval) $$
    language sql
    immutable
    returns null on null input;
EOF

-- Add a to the given timestamp the given number of days
-- Arguments:
--   $1: The timestamp
--   $2: The number of days to add / subtract (when negative)
-- Returns: The new timestamp
create or replace function cy_add_days 
    (timestamp with time zone, integer) 
    returns timestamp with time zone
    as $$ select cy_add_interval($1, $2, 'day') $$
    language sql
    immutable
    returns null on null input;
EOF

-- Returns the UNIX timestamp from a given timestamp
-- Arguments:
--   $1: The timestamp
-- Returns: The UNIX timestamp
create or replace function cy_unix_timestamp 
    (timestamp with time zone) 
    returns bigint 
    as $$ select cast(extract(epoch from $1) * 1000 as bigint) $$
    language sql
    immutable
    returns null on null input;
EOF

-- INTERNAL. Calculates the account balance difference in a given time frame
-- Arguments:
--   $1: Account id
--   $2: Begin date, may be null, assuming no limit
--   $3: End date, may be null, assuming no limit
-- Returns: The balance diff, or zero when no changes
create or replace function cy_balance_diff 
    (bigint, timestamp with time zone default null, timestamp with time zone default null)
    returns decimal as $$
        select i.amount - o.amount
        from (select coalesce(sum(amount), 0) as amount
            from transfers
            where from_id = $1
            and ($2 is null or date >= $2)
            and ($3 is null or date < $3)) o,
            (select coalesce(sum(amount), 0) as amount
            from transfers
            where to_id = $1
            and ($2 is null or date >= $2)
            and ($3 is null or date < $3)) i
    $$ language sql
    stable;
EOF

-- INTERNAL. Calculates the account reserved amount difference in a given time frame
-- Arguments:
--   $1: Account id
--   $2: Begin date, may be null, assuming no limit
--   $3: End date, may be null, assuming no limit
-- Returns: The reserved amount diff, or zero when no changes
create or replace function cy_reserved_amount_diff
    (bigint, timestamp with time zone default null, timestamp with time zone default null)
    returns decimal as $$
        select coalesce(sum(amount), 0)
        from amount_reservations
        where account_id = $1
          and ($2 is null or date >= $2)
          and ($3 is null or date < $3)
    $$ language sql
    stable;
EOF

-- Returns the current account balance and reserved amounts ignoring the account_balances / dirty_account_balances table.
-- It takes the last closed balance and sums it with the cy_balance_diff and cy_reserved_amount_diff functions.
-- Arguments:
--   p_id: The account id
--   p_date: The balance date
--   p_fetch_balance: Whether to return the balance. Defaults to true.
--   p_fetch_reserved: Whether to return the reserved amount. Defaults to true.
create type cy_account_status_type as (balance decimal, reserved decimal);
create or replace function cy_account_status_from_closing
    (p_id bigint, p_date timestamp with time zone default null, p_fetch_balance boolean default true, p_fetch_reserved boolean default true)
    returns cy_account_status_type
    as $$
        declare
            last_date timestamp with time zone;
            last_balance decimal;
            last_reserved decimal;
            v_archiving_date timestamp with time zone;
            result cy_account_status_type;
        begin
	        select archiving_date from application
	        into v_archiving_date;
	        if p_date < v_archiving_date then
	            raise exception 'BEFORE_ARCHIVING';
            end if;
            
            select c.date, c.balance, c.reserved
            from closed_account_balances c
            where c.account_id = p_id
            and (p_date is null or c.date <= p_date)
            order by date desc
            limit 1
            into last_date, last_balance, last_reserved;
            
            if last_date is null then
                -- No closed balances. Fallback to the archived balance, if any
                select archiving_date, archived_balance, archived_reserved
                from accounts
                where id = p_id
                into last_date, last_balance, last_reserved;                
            end if;
            
            if p_fetch_balance then
                result.balance := coalesce(last_balance, 0) + cy_balance_diff(p_id, last_date, p_date);
            end if;
            if p_fetch_reserved then
                result.reserved := coalesce(last_reserved, 0) + cy_reserved_amount_diff(p_id, last_date, p_date);
            end if;
            
            return result;
        end;
    $$ language plpgsql
    stable;
EOF

-- Updates the account balance and the reserved amount of a given account. 
-- Should only be invoked if the thread updating it got the lock over this specific account. It does the following:
-- 1. Do an upsert to update/insert the account_balance row for that account.
-- 2. Updates the account's negative_since if the balance was and is no longer negative, or the other way around
-- 3. Deletes all the given dirty_transfer_ids
-- Arguments:
--   p_id: Account id
--   p_balance: The new balance (already took into account the transfers referenced in p_dirty_transfer_ids)
--   p_reserved: The new reserved amount
--   p_dirty_transfer_ids: Ids of dirty transfers which were took into account to calculate the previous balance,
--     encoded as a comma-separated string (no spaces)
-- Returns: A single-row table
create or replace function cy_update_account_status(p_id bigint, p_balance decimal, p_reserved decimal, p_dirty_transfer_ids text)
    returns table(
        account_id bigint, balance decimal, negative_since_changed boolean, negative_since timestamp with time zone)
    as $$
        declare
            v_dirty_transfer_ids bigint[];
            v_negative_since timestamp with time zone;
            reserved decimal;
        begin
            account_id := p_id;
            balance := p_balance;
            
            -- Upsert the account_balance
            insert into account_balances (account_id, balance, reserved)
                values (p_id, p_balance, p_reserved)
            on conflict on constraint account_balances_pkey do update set balance = p_balance, reserved = p_reserved;
            
            -- Check if the negative since date has changed
            select a.negative_since into v_negative_since
                from accounts a
                where a.id = p_id;
            if p_balance >= 0 and v_negative_since is not null then
                -- The balance is positive / zero and we have a date - change to null
                negative_since_changed := true;
                negative_since := null;
                v_negative_since := null;
            elsif p_balance < 0 and v_negative_since is null then
                -- The balance is negative, but we have no date set - set to current date
                negative_since_changed := true;
                negative_since := now();
                v_negative_since := negative_since;
            else
                -- The current date is ok
                negative_since_changed := false;
                negative_since = v_negative_since;
            end if;
            if negative_since_changed then
                update accounts
                    set negative_since = v_negative_since
                    where id = p_id;
            end if;
            
            -- Delete the dirty account balances
            if p_dirty_transfer_ids is not null and length(p_dirty_transfer_ids) > 0 then
                v_dirty_transfer_ids := regexp_split_to_array(p_dirty_transfer_ids, ',')::bigint[];
                delete from dirty_account_balances
                    where dirty_account_balances.account_id = p_id
                    and dirty_account_balances.transfer_id = any(v_dirty_transfer_ids);
            end if;
            return next;
        end;
    $$ language plpgsql
    volatile;
EOF

-- Returns the current account status, which comprises the balance and the reserved amount.
-- Arguments:
--   p_id: Account id
--   p_reserved: Include reserved amount? If not the reserved amount will always return null. Defaults true.
-- Returns: A table a single record with the balance and the reserved amount (if requested).
--  Also returns whether the balance was cached (from account_balances) and 
--  which transfers were dirty for this balance (null if not dirty)
create or replace function cy_current_account_status
    (p_id bigint, p_reserved boolean default true)
    returns table(balance decimal, reserved decimal, cached boolean, dirty_transfer_ids bigint[])
    as $$
        declare
            last_date timestamp with time zone;
            last_balance decimal;
            balance_diff decimal;
            reserved_diff decimal;
            last_reserved decimal;
        begin
            -- Start with the cached balance
            select c.balance, c.reserved, c.dirty_transfer_ids
            from cached_account_status c
            where c.account_id = p_id
            into balance, reserved, dirty_transfer_ids;
            cached := balance is not null;
            
            if not cached then
                -- There is no account_balance for this account.
                -- Fetch the last closed balance and reserved amount
                select c.date, c.balance, c.reserved
                from last_closed_account_balances c
                where c.account_id = p_id
                into last_date, last_balance, last_reserved;
        
                -- Sum the transfers joining with dirty to fetch them in a single query.
                -- This avoids reading either ones with a committed row in between.
                -- For this reason we cannot use the cy_account_status_from_closing function here.
                select
                    coalesce(d.balance, 0) + coalesce(c.balance, 0),
                    coalesce(d.dirty_ids, array[]::bigint[]) || coalesce(c.dirty_ids, array[]::bigint[])
                from (
                    select 
                        -sum(t.amount) as balance,
                        (select array_agg(id) from unnest(array_agg(d.transfer_id)) id where id is not null) as dirty_ids                    
                    from transfers t left join dirty_account_balances d on t.id = d.transfer_id and d.account_id = p_id
                    where t.from_id = p_id
                    and (last_date is null or t.date >= last_date)
                ) as d,
                (
                    select 
                        sum(t.amount) as balance,
                        (select array_agg(id) from unnest(array_agg(d.transfer_id)) id where id is not null) as dirty_ids                    
                    from transfers t left join dirty_account_balances d on t.id = d.transfer_id and d.account_id = p_id
                    where t.to_id = p_id
                    and (last_date is null or t.date >= last_date)
                ) as c
                into balance_diff, dirty_transfer_ids;
                balance := coalesce(last_balance, 0) + balance_diff;
                
                -- Fetch the reserved amount if needed
                if p_reserved then
                    reserved := coalesce(last_reserved, 0) + cy_reserved_amount_diff(p_id, last_date);
                end if;
            end if;
            return next;
        end;
    $$ language plpgsql
    stable;
EOF

-- Calculates the account balance at a given time point.
-- If p_date is null, returns the current balance. Otherwise, a historical balance.
-- Arguments:
--   p_id: Account id
--   p_date: The balance date
-- Returns: The account balance at the given date
create or replace function cy_account_balance
    (p_id bigint, p_date timestamp with time zone default null)
    returns decimal as $$
    declare
        result decimal;
    begin
        if p_date is null then
            select c.balance
            from cy_current_account_status(p_id) c
            into result;
        else
            select c.balance
            from cy_account_status_from_closing(p_id, p_date, true, false) c
            into result;
        end if;
        return result;
    end;
    $$ language plpgsql
    stable;
EOF

-- Creates a closed_account_balance for each account that had transfers or reservations 
-- since last closed balance up to 1 hour ago.
-- Returns the number of inserted rows in closed_balances.
create or replace function cy_close_account_balances()
    returns bigint
    as $$
        declare
            v_date timestamp with time zone;
            v_result bigint;
        begin
            -- Assume a past date to avoid problems with concurrent transfers.
            -- For example, if there's a clockskew in a Java node of 1 minute,
            -- it could insert a past transfers after the balance is closed.
            v_date := now() - interval '1 hour';
            
            -- Insert the closed balances with the differences since the last closed
            insert into closed_account_balances (date, account_id, balance, reserved)
			select
			    v_date,
			    a.id,
			    coalesce(b.balance, a.archived_balance, 0) - d.amount + c.amount as balance,
			    coalesce(b.reserved, a.archived_reserved, 0) + r.amount as reserved
			from accounts a
			left join lateral (
			    select date, balance, reserved
			    from closed_account_balances b
			    where b.account_id = a.id
			    order by b.date desc
			    limit 1) b on true
			left join lateral (
			    select coalesce(sum(t.amount), 0) as amount
			    from transfers t
			    where t.from_id = a.id
			    and t.date >= coalesce(b.date, a.archiving_date, '0001-01-01'::timestamptz)
			    and t.date < v_date) d on true
			left join lateral (
			    select coalesce(sum(t.amount), 0) as amount
			    from transfers t
			    where t.to_id = a.id
			    and t.date >= coalesce(b.date, a.archiving_date, '0001-01-01'::timestamptz)
			    and t.date < v_date) c on true
			left join lateral (
			    select coalesce(sum(r.amount), 0) as amount
			    from amount_reservations r
			    where r.account_id = a.id
			    and r.date >= coalesce(b.date, a.archiving_date, '0001-01-01'::timestamptz)
			    and r.date < v_date) r on true
			where ((c.amount - d.amount) <> 0 or r.amount <> 0);  
                                
            -- Return the number of closed accounts
            get diagnostics v_result = row_count;
                        
            return v_result;
        end;
    $$ language plpgsql
    volatile;
EOF

-- Removes any account_balances and dirty_account_balances for accounts whose balance or reserved amount are inconsistent
-- Arguments:
--   p_ids: Specific account ids to fix. If null or zero length, fixes all accounts. 
-- Returns information of inconsistencies found    
create or replace function cy_fix_inconsistent_account_balances
    (variadic p_ids bigint[] default null)
    returns table(
        account_id bigint, network_id bigint,
        actual_balance numeric, expected_balance numeric,
        actual_reserved numeric, expected_reserved numeric)
    as $$
        begin
            -- Find the accounts with inconsistent balances
            drop table if exists invalid_balances;
            create temporary table invalid_balances on commit drop as
            select cached.account_id,
                cached.balance + coalesce(dirty.balance_diff, 0) as actual_balance,
                dirty.dirty_transfer_ids,
                cached.reserved as actual_reserved,
                coalesce(closed.balance, archived.balance, 0) + i.amount - o.amount as expected_balance,
                coalesce(closed.reserved, archived.reserved, 0) + r.amount as expected_reserved
            from account_balances cached
            left join (
                select d.account_id,
                    sum(case when t.from_id = d.account_id then -t.amount else amount end) as balance_diff,
                    array_agg(t.id) as dirty_transfer_ids
                from dirty_account_balances d inner join transfers t on d.transfer_id = t.id
                group by 1
            ) dirty on cached.account_id = dirty.account_id
            left join lateral (
                select date, balance, reserved
                from closed_account_balances cab 
                where cab.account_id = cached.account_id
                order by date desc
                limit 1
            ) closed on true
            left join lateral (
                select archiving_date as date, archived_balance as balance, archived_reserved as reserved
                from accounts a 
                where a.id = cached.account_id
                and closed.date is null
            ) archived on true
            inner join lateral (
                select coalesce(sum(amount), 0) as amount
                from transfers t
                where t.from_id = cached.account_id
                and t.date >= coalesce(closed.date, archived.date, '0001-01-01'::date)
            ) o on true
            inner join lateral (
                select coalesce(sum(amount), 0) as amount
                from transfers t
                where t.to_id = cached.account_id
                and t.date >= coalesce(closed.date, archived.date, '0001-01-01'::date)
            ) i on true
            inner join lateral (
                select coalesce(sum(amount), 0) as amount
                from amount_reservations ar
                where ar.account_id = cached.account_id
                and ar.date >= coalesce(closed.date, archived.date, '0001-01-01'::date)
            ) r on true
            where (p_ids is null or array_length(p_ids, 1) = 0 or cached.account_id = any(p_ids))            
                and (cached.balance + coalesce(dirty.balance_diff, 0) <> coalesce(closed.balance, archived.balance, 0) + i.amount - o.amount
                    or cached.reserved <> coalesce(closed.reserved, archived.reserved, 0) + r.amount);
            
            -- Remove the cached balances for those accounts
            delete from account_balances b
            where b.account_id in (select i.account_id from invalid_balances i);
            delete from dirty_account_balances d
                where exists (
                    select 1
                    from invalid_balances i
                    where d.account_id = i.account_id
                    and d.transfer_id = any (i.dirty_transfer_ids)
                );
                
            -- Insert the account balances for the accounts with the invalid ones
            insert into account_balances (account_id, balance, reserved)
            select i.account_id, i.expected_balance, i.expected_reserved
            from invalid_balances i 
            on conflict do nothing;
            
            -- Now return the inconsistencies found
            return query
                select
                    i.account_id, c.network_id,
                    i.actual_balance, i.expected_balance,
                    i.actual_reserved, i.expected_reserved
                from invalid_balances i
                    inner join accounts a on i.account_id = a.id
                    inner join account_types at on a.account_type_id = at.id
                    inner join currencies c on at.currency_id = c.id
                    left join users u on a.user_id = u.id
                order by u.display_for_managers nulls first;
        end;
    $$ language plpgsql
    volatile;
EOF

-- Rebuilds all closed account balances from the very beginning, generating a closed_account_balance row per day with differences until 00:00 today
-- Arguments:
--   p_ids: Specific account ids to rebuild. If null or zero length, rebuilds balaces of all accounts. 
-- Returns the number of affected accounts (accounts with transfers / reservations)
create or replace function cy_rebuild_closed_account_balances(variadic p_ids bigint[] default null)
    returns bigint
    as $$
        declare
            v_id bigint;
            v_date date;
            v_archiving_date timestamp with time zone;
            v_archived_balance decimal;
            v_archived_reserved decimal;
            v_result bigint;
        begin
            if array_length(p_ids, 1) = 0 then
                p_ids := null;
            end if;
            
            -- Remove any existing balances for those accounts
            delete from closed_account_balances where (p_ids is null or account_id = any(p_ids));
            delete from account_balances where (p_ids is null or account_id = any(p_ids));
            delete from dirty_account_balances where (p_ids is null or account_id = any(p_ids));
            
            -- Truncate the time, but taking 1 hour of tolerance for Java clockskew.
            -- The effect is: if running up to 01:00, no transfers from yesterday will be taken into account, only the day before it.
            v_date = (now() - interval '1 hour')::date;
            
            v_result := 0;

            -- Query accounts in id range which ever had transfers or reservations
            for v_id, v_archiving_date, v_archived_balance, v_archived_reserved in
                select id, archiving_date, coalesce(archived_balance, 0), coalesce(archived_reserved, 0)
                from accounts a
                where (p_ids is null or a.id = any(p_ids)) 
                and (exists (
                    select 1
                    from transfers f
                    where f.from_id = a.id
                    and (a.archiving_date is null or f.date >= a.archiving_date) 
                ) or exists (
                    select 1
                    from transfers t
                    where t.to_id = a.id
                    and (a.archiving_date is null or t.date >= a.archiving_date)
                ) or exists (
                    select 1
                    from amount_reservations r
                    where r.account_id = a.id
                    and (a.archiving_date is null or r.date >= a.archiving_date)
                ))
            loop
                v_result := v_result + 1;
                
                -- Create a row in closed_account_balances per diff
                insert into closed_account_balances (account_id, date, balance, reserved)
                select distinct *
                from (
                    select v_id, date + interval '1 day', 
                        v_archived_balance + sum(balance_diff) over (order by date), 
                        v_archived_reserved + sum(reserved_diff) over (order by date)                    
                    from (
                        select t.date::date as date, -sum(t.amount) as balance_diff, 0 as reserved_diff
                        from transfers t
                        where t.from_id = v_id
                        and (v_archiving_date is null or t.date >= v_archiving_date) 
                        and t.date < v_date
                        group by t.date::date
                        union
                        select t.date::date, sum(t.amount), 0
                        from transfers t
                        where t.to_id = v_id
                        and (v_archiving_date is null or t.date >= v_archiving_date)
                        and t.date < v_date
                        group by t.date::date
                        union
                        select r.date::date, 0, sum(r.amount)
                        from amount_reservations r
                        where r.account_id = v_id
                        and (v_archiving_date is null or r.date >= v_archiving_date)
                        and r.date < v_date
                        group by r.date::date
                    ) d
                    group by v_id, date, balance_diff, reserved_diff
                    order by date
                ) b;
            end loop;
            
            -- Update the reserved amounts for accounts which have been archived.
            -- The reservations before archiving either:
            -- * Have already been canceled (reserved and returned) and sum zero; or
            -- * Belong to some 'live' data that was not archived.
            -- So we can recalculate it here.
            update accounts
            set archived_reserved = (
                select coalesce(sum(amount), 0)
                from amount_reservations
                where account_id = accounts.id
                and date < accounts.archiving_date
            ) where archiving_date is not null
            and archived_reserved <> (
                select coalesce(sum(amount), 0)
                from amount_reservations
                where account_id = accounts.id
                and date < accounts.archiving_date
            )
            and (p_ids is null or id = any(p_ids));
            
            -- Insert the account balances for accounts
            insert into account_balances (account_id, balance, reserved)
            select a.id, b.balance, b.reserved
            from accounts a inner join lateral (
                select * from cy_account_status_from_closing(a.id)
            ) b on true
            where (p_ids is null or a.id = any(p_ids))
            on conflict do nothing;
            
            return v_result;
        end;
    $$ language plpgsql
    volatile;
EOF

-- Updates the archived date / balances for accounts created before the application.archiving_date.
-- Don't actually remove transfers, transactions or closed balances. It is the responsability
-- of an external application to actually delete those after backing up data.
-- Returns the number of affected accounts (accounts created before the given date)
create or replace function cy_archive_account_balances()
    returns bigint
    as $$
        declare
            v_date timestamp with time zone;
            v_result bigint;
        begin
	        select archiving_date
	        from application
	        into v_date;
	        
	        if v_date is null then
	           -- Nothing to archive
	           v_result = 0;
	        else
				with a as (
				    select
				        a.id, 
				        coalesce(b.balance, a.archived_balance, 0) - d.amount + c.amount as balance,
				        coalesce(b.reserved, a.archived_reserved, 0) + r.amount as reserved
				    from accounts a
				    left join lateral (
				        select date, balance, reserved
				        from closed_account_balances b
				        where b.account_id = a.id
				        and b.date < v_date
				        order by b.date desc
				        limit 1) b on true
				    left join lateral (
				        select coalesce(sum(t.amount), 0) as amount
				        from transfers t
				        where t.from_id = a.id
				        and t.date >= coalesce(b.date, a.archiving_date, '0001-01-01'::timestamptz)
				        and t.date < v_date) d on true
				    left join lateral (
				        select coalesce(sum(t.amount), 0) as amount
				        from transfers t
				        where t.to_id = a.id
				        and t.date >= coalesce(b.date, a.archiving_date, '0001-01-01'::timestamptz)
				        and t.date < v_date) c on true
				    left join lateral (
				        select coalesce(sum(r.amount), 0) as amount
				        from amount_reservations r
				        where r.account_id = a.id
				        and r.date >= coalesce(b.date, a.archiving_date, '0001-01-01'::timestamptz)
				        and r.date < v_date) r on true
				    where creation_date < v_date
				        and (archiving_date is null or archiving_date < v_date)
				)
				update accounts set
				    archiving_date = v_date,
				    archived_balance = a.balance,
				    archived_reserved = a.reserved
				from a
				where accounts.id = a.id
				and creation_date < v_date
				    and (archiving_date is null or archiving_date < v_date);
	            
	            get diagnostics v_result = row_count;
	        end if;
            return v_result;
        end;
    $$ language plpgsql
    volatile;
EOF

-- Calculates the balance sum, per time point, in a given account
-- Either positive or negative balance is used, with the given freebase (subtracted from the balance at each time point)
-- Arguments:
--   p_id: Account id
--   p_initial_balance_diff: A delta to apply on the initial balance, for the sake of calculations 
--       (for example, when calculating the reserved amount over multiple periods, need to take the 
--       previous amounts into account, but they don't yet exist as transfers)
--   p_timepoints: A comma-separated string with each timestamp
--   p_positive: Either consider positive balance (true) or negative balance (false)
--   p_freebase: The freebase to subtract from each day balance
-- Returns: The balance sum
create or replace function cy_balance_sum
    (p_id bigint, p_initial_balance_diff decimal, p_timepoints text, p_positive boolean, p_freebase decimal)
    returns decimal as $$
        declare
            v_balance decimal;
            v_balance_diff decimal;
            v_timepoints text[];
            v_timepoint text;
            v_date timestamp with time zone;
            v_previous_date timestamp with time zone;
            v_sum decimal;
            v_with_freebase decimal;
        begin
            v_balance := null;
            v_sum := 0;

            -- Split the timepoints into an array
            v_timepoints := string_to_array(p_timepoints, ',');
            
            foreach v_timepoint in array v_timepoints loop
                v_date := v_timepoint::timestamp with time zone;
                
                if v_balance is null then
                    -- First time - get the initial balance
                    select cy_account_balance(p_id, v_date) into v_balance;
                    v_balance := v_balance + p_initial_balance_diff;
                else
                    -- Get the balance diff
                    -- The balance_diff is exclusive in the begin date
                    v_balance_diff := cy_balance_diff(p_id, v_previous_date, v_date);
                    
                    v_balance := v_balance + v_balance_diff;

                    -- Apply the free base
                    if p_positive then
                        v_with_freebase := v_balance - p_freebase;
                    else
                        v_with_freebase := -v_balance - p_freebase;
                    end if;

                    -- Only update the sum if ok with the freebase
                    if v_with_freebase > 0 then
                        v_sum := v_sum + v_with_freebase;
                    end if;
                end if;
                
                v_previous_date := v_date;
            end loop;

            return v_sum;
        end;
    $$ language plpgsql
    stable;
EOF

-- Returns the hierarchical path of names on a self-referencing table through a parent identifier column
-- Arguments:
--   p_id: The element id
--   p_table: The table name
--   p_name_col: The column which contains the element name. Optional, defaults to 'name'
--   p_parent_id_col: The column which contains the id of the parent element. Optional, defaults to 'parent_id'
-- Returns: The path as string, separating names with ' > '
create or replace function cy_name_hierarchy
    (p_id bigint, p_table varchar, p_name_col varchar default 'name', p_parent_id_col varchar default 'parent_id')
    returns varchar
    as $$
        declare
            sql text;
            current_id bigint;
            v_name varchar;
            v_parent_id bigint;
            path varchar[];
        begin
            current_id := p_id;
            while current_id is not null loop
                sql := 'select id, ' || p_name_col || ', ' || p_parent_id_col
                     || ' from ' || p_table || ' where id = ' || current_id;
                execute sql into current_id, v_name, v_parent_id;
                path := array_prepend(cast(concat('_', v_name, '_') as varchar), path);
                current_id := v_parent_id;
            end loop;
            return array_to_string(path, 'a');
        end;
    $$ language plpgsql
    stable;
EOF

-- Returns the hierarchical path of orders on a self-referencing table through a parent identifier column.
-- Orders are left padded, so it can be safely used to sort results hierarchically
-- Arguments:
--   p_id: The element id
--   p_table: The table name
--   p_name_col: The column which contains the element name. Optional, defaults to 'name'
--   p_parent_id_col: The column which contains the id of the parent element. Optional, defaults to 'parent_id'
-- Returns: The path as string, separating names with ' > '
create or replace function cy_order_hierarchy
    (p_id bigint, p_table varchar, p_order_col varchar default 'order_index', p_parent_id_col varchar default 'parent_id')
    returns varchar
    as $$
        declare
            sql text;
            current_id bigint;
            v_order varchar;
            v_parent_id bigint;
            orders varchar[];
        begin
            current_id := p_id;
            while current_id is not null loop
                sql := 'select id, lpad(cast (' || p_order_col || ' as varchar), 9, ''0''), ' || p_parent_id_col
                     || ' from ' || p_table || ' where id = ' || current_id;
                execute sql into current_id, v_order, v_parent_id;
                orders := array_prepend(v_order, orders);
                current_id := v_parent_id;
            end loop;
            return array_to_string(orders, ' > ');
        end;
    $$ language plpgsql
    stable;
EOF

-- Removes all HTML tags, replacing them by a space. A space is needed, so cases like '<div>word1</div>word2'
-- would be returned as 'word1word2', not 'word1 word2'
create or replace function cy_strip_html_tags
    (text)
    returns text
    as $$ select regexp_replace($1, '<[^>]*>', ' ', 'g'); $$
    language sql
    immutable
    returns null on null input;
EOF

-- Returns a geography type for the given latitude and longitude
create or replace function cy_to_geography
    (float8, float8)
    returns geography
    as $$ select ST_GeographyFromText(concat('POINT(', $2, ' ', $1, ')')); $$
    language sql immutable;
EOF

-- Returns all tsvectors on addresses for a given user, and optionally including those which are hidden
create or replace function cy_get_addresses_tsvector
    (p_user_id bigint, p_include_hidden boolean)
    returns tsvector as $$
        declare
            v_tsvector tsvector;
            v_result tsvector;
        begin
            v_result = to_tsvector('');
            for v_tsvector in
                select full_tsvector
                from addresses
                where user_id = p_user_id
                  and (p_include_hidden is true or hidden is false)
            loop
                if v_tsvector is not null then
                    v_result := v_result || v_tsvector;
                end if;
            end loop;
            return v_result;
        end;
    $$ language plpgsql
    stable;
EOF

-- Re-index all data in the given network
create or replace function cy_reindex_network
    (p_network_id bigint)
    returns void 
    as $$
        begin
            -- Reindex users
            update users set
                name_tsvector = setweight(to_tsvector_net(p_network_id, name), 'B')
                where network_id = p_network_id or p_network_id is null and network_id is null;
                
            -- Reindex addresses 
            update addresses set
                full_tsvector = to_tsvector_net(p_network_id, concat(address_line_1, ' ', address_line_2, ' ', street, ' ', building_number, ' ', complement, ' ',  city, ' ', neighborhood, ' ', po_box, ' ', region, ' ', zip)),
                address_tsvector = to_tsvector_net(p_network_id, concat(address_line_1, ' ', address_line_2, ' ', street, ' ', building_number, ' ', complement)),
                neighborhood_tsvector = to_tsvector_net(p_network_id, neighborhood),
                city_tsvector = to_tsvector_net(p_network_id, city),
                region_tsvector = to_tsvector_net(p_network_id, region)
                where user_id in (select id from users where network_id = p_network_id or p_network_id is null and network_id is null);
                
            -- Reindex ads 
            update ads set
                name_tsvector = setweight(to_tsvector_net(p_network_id, name), 'A'),
                description_tsvector = setweight(to_tsvector_net(p_network_id, description), 'B')
                where owner_id in (select id from users where network_id = p_network_id or p_network_id is null and network_id is null);
            
            -- Reindex user custom field values
            update user_custom_field_values set
                value_tsvector = to_tsvector_net(p_network_id, coalesce(string_value, text_value, rich_text_value, ''))
                where owner_id in (select id from users where network_id = p_network_id or p_network_id is null and network_id is null);
            
            -- Reindex ad custom field values
            update ad_custom_field_values set
                value_tsvector = to_tsvector_net(p_network_id, coalesce(string_value, text_value, rich_text_value, ''))
                where owner_id in (select a.id from ads a inner join users u on a.owner_id = u.id where u.network_id = p_network_id or p_network_id is null and u.network_id is null);
            
            -- Reindex record custom field values
            update record_custom_field_values set
                value_tsvector = to_tsvector_net(p_network_id, coalesce(string_value, text_value, rich_text_value, ''))
                where owner_id in (select r.id from records r inner join record_types rt on r.type_id=rt.id where rt.network_id = p_network_id or p_network_id is null and rt.network_id is null);
        end;
    $$ language plpgsql;
EOF

-- Dispatches external indexing of data via a background task
create or replace function cy_insert_tasks_reindex
    (class_name text, index_name text, table_name text, batch_size integer, condition text default '')
    returns bigint 
    as $$
        declare
            sql text;
            affected_rows bigint;
        begin
            sql := 'insert into background_task_executions (class_name, context, priority, display) '
                || format('select ''%s'', concat(''%s|'', string_agg(id::text, '','')), 0, ''Indexing of %s'' ', class_name, index_name, table_name)
                || 'from ( '
                || ' select id, row_number() over (order by id) as n '
                || ' from ' || table_name
                || ' ' || condition
                || ') x group by n / ' || batch_size;
            execute sql;
            GET DIAGNOSTICS affected_rows = ROW_COUNT;
            return affected_rows;
        end; 
    $$ language plpgsql volatile;
EOF

-- Dispatches entity notifications processing via a background task
create or replace function cy_insert_tasks_entity_notifications
    (class_name text, entity_type text, table_name text, batch_size integer)
    returns bigint 
    as $$
        declare
            sql text;
            temp_table_name text;
            affected_rows bigint;
        begin
            temp_table_name = table_name || '_to_notify';
            sql := 'drop table if exists ' || temp_table_name;
            execute sql;
            sql := 'create temp table ' || temp_table_name || ' on commit drop as select id from ' || table_name || ' where pending_notification is true';
            execute sql;
            sql := 'insert into background_task_executions (class_name, context, priority, display) '
                || format('select ''%s'', concat(''%s|'', string_agg(id::text, '','')), 1, ''Dispatch notifications of %s'' ', class_name, entity_type, table_name)
                || 'from ( '
                || ' select id, row_number() over (order by id) as n '
                || ' from ' || temp_table_name
                || ') x group by n / ' || batch_size;
            execute sql;
            sql := 'update ' || table_name || ' set pending_notification = false where id in (select id from ' || temp_table_name || ')';
            execute sql;
            GET DIAGNOSTICS affected_rows = ROW_COUNT;
            return affected_rows;
        end; 
    $$ language plpgsql volatile;
EOF

-- Updates an icon name in all places it is used
create or replace function cy_replace_icon
    (old_icon text, new_icon text)
    returns table(table_name text, updated integer) 
    as $$
        declare
            c record;
        begin
            for c in
	            select col.table_name
	            from information_schema.columns col
	            where col.table_schema = 'public'
	            and col.column_name = 'svg_icon'
            loop
                execute concat('update ', c.table_name, ' set svg_icon = ''', new_icon, ''' where svg_icon = ''', old_icon, '''');
                table_name := c.table_name;
                get diagnostics updated = row_count;
                return next;
            end loop;
        end; 
    $$ language plpgsql volatile;
EOF

-- Creates the given constraint if not exists
-- Arguments:
--   $1: The table name where the constraint will be added
--   $2: The constraint name 
--   $3: The constraint definition
create or replace function cy_create_constraint_if_not_exists 
    (t_name text, c_name text, constraint_sql text)
    returns void
    as $$
    begin
        -- Look for our constraint
        if not exists (select constraint_name
                       from information_schema.table_constraints
                       where table_name = t_name  and constraint_name = c_name) then
            execute 'ALTER TABLE ' || t_name || ' ADD CONSTRAINT ' || c_name || ' ' || constraint_sql;
        end if;
    end; $$
    language plpgsql volatile;
EOF

-- Revert the delimiter
DELIMITER ;